/*
 *
 * Sony Programmable I/O Control Device driver for VAIO. 
 * Userspace X11 Daemon Utility
 *
 * Original Copyright 2001 Stelian Pop, Alcove
 * Modified by Akkana, http://www.shallowsky.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include <X11/Xlib.h>
#include <X11/extensions/XTest.h>

#include <sys/ioctl.h>

#include <linux/sonypi.h>

static int Debug = 0;

static void inline simulateKeyPress(Display *disp, char *keyname) {
	KeyCode keycode = 0;
	keycode = XKeysymToKeycode(disp, XStringToKeysym(keyname));
	XTestGrabControl(disp, True);
	XTestFakeKeyEvent(disp, keycode, True, 0);
	XSync(disp, False);
	XTestGrabControl(disp, False);
}

static void inline simulateKeyRelease(Display *disp, char *keyname) {
	KeyCode keycode = 0;
	keycode = XKeysymToKeycode(disp, XStringToKeysym(keyname));
	XTestGrabControl(disp, True);
	XTestFakeKeyEvent(disp, keycode, False, 0);
	XSync(disp, False);
	XTestGrabControl(disp, False);
}

static void inline simulateButton(Display *disp, int button) {
	XTestGrabControl(disp, True);
	XTestFakeButtonEvent(disp, button, True, 0);
	XTestFakeButtonEvent(disp, button, False, 0);
	XSync(disp, False);
	XTestGrabControl(disp, False);
}

static void inline verboseEvent(char *event) {
        if (Debug)
	    fprintf(stderr, "Event: %s\n", event);
}

static int spic_ioctl(int, __u8 *);
void changeBrightness(int incr);

/*
 * Main function.
 */
int main(int argc, char *argv[]) {
	int fd;
	fd_set rfds;
	int retval;
	unsigned char event;
	Display *disp = NULL;

	if (argc > 1 && argv[1][0] == '-' && argv[1][1] == 'd')
	    Debug = 1;

	disp = XOpenDisplay(NULL);
	if (!disp) {
		fprintf(stderr, "Failed to open display\n");
		return -1;
	}

opendev:
	if ( (fd = open("/dev/sonypi", O_RDONLY)) == -1) {
		fprintf(stderr, "Failed to open /dev/sonypi: %s\n", 
			strerror(errno));
		return -1;
	}
again:
	FD_ZERO(&rfds);
	FD_SET(fd, &rfds);

	retval = select(fd + 1, &rfds, NULL, NULL, NULL);

	if (retval == -1) {
		fprintf(stderr, "select failed: %s\n", strerror(errno));
		return -1;
	}

	if (FD_ISSET(fd, &rfds)) {
                static int pressed = 0;
		if (read(fd, &event, sizeof(event)) != sizeof(event))
		{
		    fprintf(stderr, "read failed: %s\n", strerror(errno));
		    close(fd);
		    goto opendev;
		}
		switch (event) {
			case SONYPI_EVENT_JOGDIAL_DOWN:
				simulateButton(disp, 5);
				verboseEvent("Jogdial Down");
				break;
			case SONYPI_EVENT_JOGDIAL_UP:
				simulateButton(disp, 4);
				verboseEvent("Jogdial Up");
				break;
			case SONYPI_EVENT_JOGDIAL_DOWN_PRESSED:
				verboseEvent("Jogdial Down & Pressed");
                                changeBrightness(-12);
                                pressed = 1;
				break;
			case SONYPI_EVENT_JOGDIAL_UP_PRESSED:
				verboseEvent("Jogdial Up & Pressed");
                                changeBrightness(12);
                                pressed = 1;
				break;
			case SONYPI_EVENT_JOGDIAL_PRESSED:
				verboseEvent("Jogdial Pressed");
				break;
			case SONYPI_EVENT_JOGDIAL_RELEASED:
				verboseEvent("Jogdial Released");
                                if (!pressed)
                                    simulateButton(disp, 2);
                                pressed = 0;
				break;
			case SONYPI_EVENT_CAPTURE_PRESSED:
				if (!fork()) {
					execlp("xterm", "xterm", 0);
					exit(0);
				}
				verboseEvent("Capture Pressed");
				break;
			case SONYPI_EVENT_CAPTURE_RELEASED:
				verboseEvent("Capture Released");
				break;
			case SONYPI_EVENT_CAPTURE_PARTIALPRESSED:
				verboseEvent("Capture Partial Pressed");
				break;
			case SONYPI_EVENT_CAPTURE_PARTIALRELEASED:
				verboseEvent("Capture Partial Released");
				break;
			case SONYPI_EVENT_FNKEY_ESC:
				verboseEvent("Fn-Esc Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F1:
				verboseEvent("Fn-F1 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F2:
				verboseEvent("Fn-F2 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F3:
				verboseEvent("Fn-F3 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F4:
				verboseEvent("Fn-F4 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F5:
                                changeBrightness(15);
				verboseEvent("Fn-F5 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F6:
                                changeBrightness(-15);
				verboseEvent("Fn-F6 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F7:
				verboseEvent("Fn-F7 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F8:
				verboseEvent("Fn-F8 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F9:
				verboseEvent("Fn-F9 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F10:
				verboseEvent("Fn-F10 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F11:
				verboseEvent("Fn-F11 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F12:
				verboseEvent("Fn-F12 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_1:
				verboseEvent("Fn-1 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_2:
				verboseEvent("Fn-2 Pressed");
				break;
			case SONYPI_EVENT_FNKEY_D:
				verboseEvent("Fn-D Pressed");
				break;
			case SONYPI_EVENT_FNKEY_E:
				verboseEvent("Fn-E Pressed");
				break;
			case SONYPI_EVENT_FNKEY_F:
				verboseEvent("Fn-F Pressed");
				break;
			case SONYPI_EVENT_FNKEY_S:
				verboseEvent("Fn-S Pressed");
				break;
			case SONYPI_EVENT_FNKEY_B:
				verboseEvent("Fn-B Pressed");
				break;
			case SONYPI_EVENT_BLUETOOTH_PRESSED:
				verboseEvent("Bluetooth Button Pressed");
				break;
#if 0 /* 2.4.2 kernel doesn't define these symbols */
			case SONYPI_EVENT_PKEY_P1:
				verboseEvent("P1 Pressed");
				break;
			case SONYPI_EVENT_PKEY_P2:
				verboseEvent("P2 Pressed");
				break;
			case SONYPI_EVENT_PKEY_P3:
				verboseEvent("P3 Pressed");
				break;
#endif
			default:
				verboseEvent("Event unknown!");
		}
	}

	goto again;

	close(fd);
	return 0;
}

void changeBrightness(int incr)
{
    __u8 value8;
    if (spic_ioctl(SONYPI_IOCGBRT, &value8) < 0)
        return;
    if (Debug)
        printf("Change Brightness by %d: old brightness was %u, ",
               incr, (unsigned int)value8);
    value8 += incr;
    if (Debug)
        printf("setting to %u\n", (unsigned int)value8);
    spic_ioctl(SONYPI_IOCSBRT, &value8);
}

/*
 * Sends a ioctl command to the SPIC driver.
 *
 * Return: 0 if success, -1 if an error occured.
 */
static int spic_ioctl(int ioctlno, __u8 *param) {
	int fd;

	if ( (fd = open("/dev/sonypi", O_RDWR)) == -1) {
		fprintf(stderr, "Failed to open /dev/sonypi: %s\n", 
			strerror(errno));
		return -1;
	}
	if (ioctl(fd, ioctlno, param) < 0) {
		fprintf(stderr, "ioctl failed: %s\n", strerror(errno));
		return -1;
	}
	close(fd);
	return 0;
}

